<?php

namespace Sqits\Dropbox\Resources;

use Http\Discovery\Psr17FactoryDiscovery;
use Sqits\Dropbox\DataObjects\Metadata;
use Sqits\Dropbox\Enums\Method;
use Sqits\Dropbox\Exceptions\folder\FailedToFetchMetadataException;
use Sqits\Dropbox\Resources\Concerns\CanUseClient;

class MetadataResource
{
    use CanUseClient;

    /**
     * @param string $path
     * @return Metadata
     * @throws FailedToFetchMetadataException
     * @throws \JsonException
     */
    public function find(string $path): Metadata
    {
        $request = $this->buildRequest(Method::POST, 'https://api.dropboxapi.com/2/files/get_metadata')
            ->withBody(
                Psr17FactoryDiscovery::findStreamFactory()
                    ->createStream(json_encode([
                        'path' => $path,
                        'include_media_info' => false,
                        'include_deleted' => false,
                        'include_has_explicit_shared_members' => false,
                    ])));
        try {
            $response = $this->client->send($request);

            $statusCode = $response->getStatusCode();


            if (!($statusCode >= 200 && $statusCode <= 299)) {
                throw new FailedToFetchMetadataException(
                    "Failed to fetch metadata with path: {$path}",
                    $response->getStatusCode()
                );
            }

        } catch (\Throwable $exception) {
            throw new FailedToFetchMetadataException(
                "Failed to fetch metadata with path: {$path}",
                $exception->getCode(),
                $exception
            );
        }
        
        /**
         * @var array{
         *     ".tag": string,
         *     "name": string,
         *     "path_lower": string,
         *     "path_display": string,
         *     "id": string,
         *     "client_modified": string|null,
         *     "server_modified": string|null,
         *     "rev": string|null,
         *     "size": int|null,
         *     "is_downloadable": bool|null,
         *     "content_hash": string|null,
         *     "file_lock_info": array{
         *          created: string,
         *          is_lockholder: bool,
         *          lockholder_name: string
         *     }|null,
         *     "has_explicit_shared_members": bool|null,
         *     "property_groups": array{
         *         name: string,
         *         value: string
         *     }|null,
         *     "sharing_info": array{
         *          modified_by: string,
         *          parent_shared_folder_id: string,
         *          read_only: bool
         *     }|null
         * } $data
         */
        $data = json_decode(
            $response->getBody()->getContents(),
            true,
            JSON_THROW_ON_ERROR,
            JSON_THROW_ON_ERROR);


        return Metadata::fromResponse($data);
    }
}
