<?php

namespace Sqits\Dropbox\Resources;

use Http\Discovery\Psr17FactoryDiscovery;
use Sqits\Dropbox\DataObjects\File;
use Sqits\Dropbox\DataObjects\Metadata;
use Sqits\Dropbox\Enums\Method;
use Sqits\Dropbox\Exceptions\folder\FailedToDeleteFileException;
use Sqits\Dropbox\Exceptions\folder\FailedToUploadFileException;
use Sqits\Dropbox\Resources\Concerns\CanUseClient;

class FileResource
{
    use CanUseClient;

    public function upload(string $filepath, string $dropboxPath, $args = [
        'autorename' => true,
    ]): File
    {
        $request = $this->buildRequestWithArgs(
            Method::POST,
            'https://content.dropboxapi.com/2/files/upload',
            array_merge($args, [
                'path' => $dropboxPath,
            ])
        )
            ->withHeader('Content-Type', 'application/octet-stream')
            ->withBody(
                Psr17FactoryDiscovery::findStreamFactory()
                    ->createStream(file_get_contents($filepath))
            );

        try {
            $response = $this->client->send($request);

            $statusCode = $response->getStatusCode();

            if ($statusCode > 299) {
                throw new FailedToUploadFileException(
                    "Failed to upload file to Dropbox",
                    $statusCode
                );
            }

        } catch (\Throwable $exception) {
            throw new FailedToUploadFileException(
                "Failed to upload file to Dropbox",
                $exception->getCode(),
                $exception
            );
        }

        /**
         * @param array{
         *     client_modified: string,
         *     content_hash: string,
         *     has_explicit_shared_members: bool,
         *     id: string,
         *     is_downloadable: bool,
         *     name: string,
         *     path_display: string,
         *     path_lower: string,
         *     rev: string,
         *     server_modified: string,
         *     size: int,
         *     file_lock_info: array{
         *     created: ?string,
         *     is_lockholder: ?bool,
         *     lockholder_name: ?string
         *    }|null,
         *     property_groups: array{
         *     fields: array{
         *     name: string,
         *     value: string
         *   },
         *     template_id: string
         *  }|null,
         *     sharing_info: array{
         *     modified_by: string,
         *     parent_shared_folder_id: string,
         *     read_only: bool,
         *     }|null
         * } $data
         */
        $data = json_decode(
            $response->getBody()->getContents(),
            true,
            JSON_THROW_ON_ERROR,
            JSON_THROW_ON_ERROR);


        return File::fromResponse($data);
    }


    public function delete(string $path, array $args = []): Metadata
    {
        $request = $this->buildRequest(Method::POST, 'https://api.dropboxapi.com/2/files/delete_v2')
            ->withBody(
                Psr17FactoryDiscovery::findStreamFactory()
                    ->createStream(json_encode(array_merge($args, [
                        'path' => $path,
                    ]))));

        try {
            $response = $this->client->send($request);

            $statusCode = $response->getStatusCode();

            if ($statusCode > 299) {
                throw new FailedToDeleteFileException(
                    json_decode($response->getBody()->getContents(), true),
                    $statusCode
                );
            }

        } catch (\Throwable $exception) {
            throw new FailedToDeleteFileException(
                "Failed to delete file from Dropbox",
                $exception->getCode(),
                $exception
            );
        }

        /**
         * @param array{
         *     metadata: Metadata
         *      } $data
         */
        $data = json_decode(
            $response->getBody()->getContents(),
            true,
            JSON_THROW_ON_ERROR,
            JSON_THROW_ON_ERROR);

        return Metadata::fromResponse($data['metadata']);
    }
}
