<?php

namespace Flex\TranslationManager\Actions;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Lang;

class importTranslationsAction
{
    public function execute($command = null): void
    {
        $languagePath = lang_path();
        $directories = File::directories($languagePath);

        foreach ($directories as $directory) {
            $languageModel = $this->getOrCreateLanguage($directory);
            if ($command) {
                $command->info("Syncing language folder: {$directory}");
            }

            $this->syncingTranslation($directory, $languageModel);
            if ($command) {
                $command->info("Successfully synced language folder: {$directory}");
            }
        }
    }

    private function getOrCreateLanguage(mixed $directory)
    {
        $folderName = basename($directory);
        $languageModelClass = Config::get('translation-manager.translation_language_model');

        return $languageModelClass::firstOrCreate(['folder_name' => $folderName], [
            'name' => $folderName,
            'folder_name' => $folderName,
        ]);
    }

    protected function syncingTranslation(string $languageDirectory, $languageModel): void
    {
        $files = File::allFiles($languageDirectory);
        foreach ($files as $file) {
            // only sync php files
            if ($file->getExtension() !== 'php') {
                continue;
            }
            $fileName = $file->getBasename('.php');
            $tmpKeyPrefixes = [];

            $translationArray = require $file->getPathname();

            if (! is_array($translationArray) || count($translationArray) === 0) {
                continue;
            }

            $this->importTranslationsRecursively($translationArray, $fileName, $languageModel);

        }

        // Delete translations that are no longer in the language files
        $translations = $languageModel->translations()->get();
        foreach ($translations as $translation) {
            $key = "{$translation->file_name}.{$translation->key}";
            $locale = $languageModel->folder_name;
            if (! Lang::has($key, $locale)) {
                $translation->delete();
            }
        }
    }

    private function importTranslationsRecursively($data, $fileName, $languageModel): void
    {
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $data = Arr::dot($value);
                foreach ($data as $nestedKey => $nestedValue) {
                    if (is_array($nestedValue)) {
                        if (count($nestedValue) === 0) {
                            continue;
                        }
                        $this->importTranslationsRecursively($nestedValue, $fileName, $languageModel);
                    } else {
                        $newKey = "{$key}.{$nestedKey}";
                        $this->addOrUpdateTranslation($newKey, $nestedValue, $fileName, $languageModel);
                    }
                }
            } else {
                $this->addOrUpdateTranslation($key, $value, $fileName, $languageModel);
            }
        }
    }

    private function addOrUpdateTranslation($key, $value, $fileName, $languageModel)
    {
        $languageModel->translations()->updateOrCreate(
            [
                'key' => $key],
            [
                'value' => $value,
                'file_name' => $fileName,
            ]
        );
    }
}
