<?php

namespace Flex\TranslationManager\Http\Controllers\Api;

use Flex\TranslationManager\Http\Requests\StoreUpdateTranslationRequest;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class TranslationLanguageController
{
    public function index(Request $request): JsonResponse
    {
        $requestProperties = $request->validate([
            'locale' => ['required', 'string'],
        ]);

        $locale = $requestProperties['locale'];

        $languageModel = config('translation-manager.translation_language_model');
        $languageModel = $languageModel::query()->where('name', $locale)->firstOrFail();

        $translations = $languageModel
            ->translations()
            ->get()
            ->map(function ($translation) {
                return [
                    'id' => $translation->id,
                    'key' => $translation->key,
                    'display_key' => "{$translation->file_name}.{$translation->key}",
                    'value' => $translation->value,
                ];
            });

        $data = [
            'language' => $languageModel,
            'translations' => $translations,
        ];

        return response()->json($data);
    }

    public function store(Request $request): JsonResponse
    {
        try {
            $requestProperties = $request->validate([
                'locale' => ['required', 'string'],
                'translation_key' => ['required', 'string'],
                'translation_value' => ['required', 'string'],
            ]);

            $fileNameRequired = config('translation-manager.separate_translation_files');

            if (! $fileNameRequired && ! str_contains($requestProperties['translation_key'], '.')) {
                throw ValidationException::withMessages([
                    'translation_key' => 'The translation key must contain a dot (.)',
                ])->errorBag('translations');
            }

            $languageModel = config('translation-manager.translation_language_model');
            $translationModel = config('translation-manager.translation_model');

            $languageModel = $languageModel::query()->where('name', $requestProperties['locale'])->firstOrFail();

            $splitKey = explode('.', $requestProperties['translation_key']);

            $fileName = $splitKey[0];

            // remove index 0 from key
            array_shift($splitKey);

            $translationKey = implode('.', $splitKey);

            $translationModel::query()->updateOrCreate([
                'translation_language_id' => $languageModel->id,
                'key' => $translationKey,
                'file_name' => $fileName,
            ], [
                'value' => $requestProperties['translation_value'],
            ]);

            return response()->json();
        } catch (ValidationException $e) {
            return response()->json([
                'message' => $e->validator->getMessageBag()->getMessages(),
            ], 422);
        } catch (\Exception $e) {
            report($e);

            return response()->json([
                'message' => 'Error creating translation.',
            ], 422);
        }
    }

    public function update(StoreUpdateTranslationRequest $request): JsonResponse
    {
        try {
            $requestProperties = $request->validated();
            $languageModel = config('translation-manager.translation_language_model');
            $translationModel = config('translation-manager.translation_model');

            $languageModel = $languageModel::query()->where('name', $requestProperties['locale'])->firstOrFail();

            $translationModel::query()->updateOrCreate([
                'translation_language_id' => $languageModel->id,
                'key' => $requestProperties['key'],
                'file_name' => $requestProperties['file_name'],
            ], [
                'value' => $requestProperties['value'],
            ]);

            return response()->json();
        } catch (\Exception $e) {
            report($e);

            return response()->json([
                'message' => 'Error updating translation.',
            ], 422);
        }
    }

    public function destroy($id): JsonResponse
    {
        try {
            $translationModel = config('translation-manager.translation_model');

            $translationModel::query()->findOrFail($id)->delete();

            return response()->json();
        } catch (\Exception $e) {
            report($e);

            return response()->json([
                'message' => 'Error deleting translation.',
            ], 422);
        }
    }
}
