<?php

namespace Flex\TranslationManager;

use Flex\TranslationManager\Commands\InstallCommand;
use Flex\TranslationManager\Commands\PublishCommand;
use Flex\TranslationManager\Commands\TranslationSyncCommand;
use Flex\TranslationManager\Http\Middleware\Authenticate;
use Flex\TranslationManager\Observers\TranslationObserver;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Route;
use Spatie\LaravelPackageTools\Package;
use Spatie\LaravelPackageTools\PackageServiceProvider;

class TranslationManagerServiceProvider extends PackageServiceProvider
{
    public function configurePackage(Package $package): void
    {
        /*
         * This class is a Package Service Provider
         *
         * More info: https://github.com/spatie/laravel-package-tools
         */
        $package
            ->name('translation-manager')
            ->hasConfigFile()
            ->hasViews()
            ->hasMigration('create_translation_languages_table')
            ->hasMigration('create_translations_table')
            ->hasMigration('create_translation_histories_table')
//            ->hasInstallCommand(function(\Spatie\LaravelPackageTools\Commands\InstallCommand $command) {
//                $command
////                    ->publishConfigFile()
////                    ->publishMigrations()
//                    ->publishAssets();
//            })
            ->hasCommand(InstallCommand::class)
            ->hasCommand(PublishCommand::class)
            ->hasCommand(TranslationSyncCommand::class);
    }

    public function packageRegistered(): void
    {
        if (! defined('TRANSLATION_MANAGER_PATH')) {
            define('TRANSLATION_MANAGER_PATH', realpath(__DIR__.'/../'));
        }
        $this->registerRoutes();
        $this->defineAssetPublishing();
    }

    public function defineAssetPublishing(): void
    {
        if ($this->app->runningInConsole()) {
            $this->publishes([
                TRANSLATION_MANAGER_PATH.'/public' => public_path('vendor/translation-manager'),
            ], 'translation-manager-assets');
        }
    }

    public function registerRoutes(): void
    {
        Route::group([
            'domain' => config('translation-manager.domain', null),
            'prefix' => config('translation-manager.path'),
            'namespace' => 'Flex\TranslationManager\Http\Controllers',
            'middleware' => config('translation-manager.middleware', ['web', Authenticate::class]),
            'name' => 'translation-manager.',
        ], function () {
            $this->loadRoutesFrom(__DIR__.'/../routes/web.php');
            $this->loadRoutesFrom(__DIR__.'/../routes/api.php');
        });
    }

    public function bootingPackage(): void
    {
        $translationModelClass = config('translation-manager.translation_model');
        $translationModelClass::observe(TranslationObserver::class);
    }

    public function packageBooted()
    {
        $this->registerAuthorization();
    }

    protected function registerAuthorization()
    {
        Gate::define('viewTranslationManager', function ($user = null) {
            return app()->environment('local');
        });
    }
}
