<?php

namespace Flex\TranslationManager\Http\Controllers\Api;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TranslationLanguageController
{
    public function index(Request $request): JsonResponse
    {
        $locale = config('app.locale');
        if (! $locale) {
            $locale = config('app.fallback_locale');
        } elseif ($request->has('locale')) {
            $locale = $request->get('locale');
        }

        $languageModel = config('translation-manager.translation_language_model');
        if (! $locale) {
            $defaultLanguage = $languageModel::query()->first();
        } else {
            $defaultLanguage = $languageModel::query()->where('folder_name', $locale)->first();
        }

        $searchQuery = $request->search ?? '';
        $translationKey = DB::raw("CONCAT(translations.file_name, '.', translations.key)");
        $translationQuery = $defaultLanguage
            ->translations();
        if (strlen($searchQuery) > 0) {
            $translationQuery->where($translationKey, 'like', "%{$searchQuery}%");
        }

        $translations = $translationQuery->get()->map(function ($translation) {
            return [
                'id' => $translation->id,
                'key' => $translation->key,
                'display_key' => "{$translation->file_name}.{$translation->key}",
                'value' => $translation->value,
            ];
        });

        $data = [
            'language' => $defaultLanguage,
            'translations' => $translations,
        ];

        return response()->json($data);
    }

    public function update(Request $request, $id): JsonResponse
    {
        try {
            $translationModel = config('translation-manager.translation_model');

            $translationModel::query()->updateOrCreate([
                'translation_language_id' => $id,
                'key' => $request->key,
                'file_name' => $request->file_name,
            ], [
                'value' => $request->value,
            ]);

            return response()->json();
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'Error updating translation.',
            ], 422);
        }
    }
}
