<?php

namespace Flex\TranslationManager\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\LazyCollection;

class TranslationSyncCommand extends Command
{
    public $signature = 'translation-manager:sync';

    public $description = 'Sync translations';

    public function handle(): int
    {
        $languagePath = lang_path();
        $directories = File::directories($languagePath);

        foreach ($directories as $directory) {
            $languageModel = $this->getOrCreateLanguage($directory);
            $this->info("Syncing language folder: {$directory}");
            $this->syncingTranslation($directory, $languageModel);
            $this->info("Successfully synced language folder: {$directory}");
        }

        return self::SUCCESS;
    }

    /**
     * Get all the translation keys for a given locale.
     *
     * @param string $locale
     * @return array
     */
    protected function syncingTranslation(string $languageDirectory, $languageModel): void
    {
        $files = File::allFiles($languageDirectory);
        foreach ($files as $file) {
            // only sync php files
            if ($file->getExtension() !== 'php') {
                continue;
            }
            $fileName = $file->getBasename('.php');
            $tmpKeyPrefixes = [];

            $translationArray = require($file->getPathname());
            foreach ($translationArray as $key => $value) {

                $languageModel->translations()->updateOrCreate(
                    [
                        'key' => $key],
                    [
                        'value' => $value,
                        'file_name' => $fileName,
                    ]
                );
            }
//            LazyCollection::make(function () use ($file) {
//                $pathName = $file->getPathname();
//                $handle = fopen($pathName, 'r');
//                while (($line = fgets($handle)) !== false) {
//                    yield $line;
//                }
//            })
//                ->each(function ($line) use ($languageModel, &$tmpKeyPrefixes, $fileName) {
//                    $keyValue = $this->resolveLine(
//                        line: $line,
//                        languageModel: $languageModel,
//                        tmpKeyPrefixes: $tmpKeyPrefixes,
//                    );
//
//                    if (empty($keyValue)) {
//                        return;
//                    }
//
//                    $languageModel->translations()->updateOrCreate(
//                        [
//                            'key' => $keyValue['key']],
//                        [
//                            'value' => $keyValue['value'],
//                            'file_name' => $fileName,
//                        ]
//                    );
//                });
        }

        // Delete translations that are no longer in the language files
        $translations = $languageModel->translations()->get();
        foreach ($translations as $translation) {
            $key = "{$translation->file_name}.{$translation->key}";
            $locale = $languageModel->folder_name;
            if (!Lang::has($key, $locale)) {
                $translation->delete();
            }
        }
    }

    private function getOrCreateLanguage(mixed $directory)
    {
        $folderName = basename($directory);
        $languageModelClass = Config::get('translation-manager.translation_language_model');

        return $languageModelClass::firstOrCreate(['folder_name' => $folderName], [
            'name' => $folderName,
            'folder_name' => $folderName,
        ]);
    }

    private function resolveKeyOrValue(string $value): string
    {
        $value = trim($value);
        if (str_starts_with($value, "'") || str_starts_with($value, '"')) {
            $value = substr($value, 1, -1);
            if (str_ends_with($value, "'") || str_ends_with($value, '"')) {
                $value = substr($value, 0, -1);
            }
        }

        // Check if value contains escape slash and remove it
        if (str_contains($value, '\\')) {
            $value = str_replace('\\', '', $value);
        }

        return $value;
    }

    private function resolveLine(string $line, string $languageModel, array &$tmpKeyPrefixes): array
    {
        $line = trim($line);

        $valuesToIgnore = [
            '<?php',
            //            '];',
            //            '[',
            'return [',
        ];

        if (in_array($line, $valuesToIgnore) || empty($line)) {
            return [];
        }

        // Check if we are at the end of a nested array
        if ($line === ']' || $line === '],') {
            if (count($tmpKeyPrefixes) > 0) {
                array_pop($tmpKeyPrefixes);

                return [];
            }
        }

        // Check if line contains a key value pair for a translation
        if (!str_contains($line, '=>')) {
            return [];
        }

        // Check is nested array begins on new line
        if (str_contains($line, '=>') && empty(explode('=>', $line)[1])) {
            $possibleNestedArray = explode('=>', $line);
            $possibleNestedArrayKey = $this->resolveKeyOrValue($possibleNestedArray[0]);
            $possibleNestedArrayValue = $possibleNestedArray[1];
            if (empty($possibleNestedArrayValue)) {
                $tmpKeyPrefixes[] = $possibleNestedArrayKey;

                return [];
            }
        }

        // Get the key and value
        [$key, $value] = explode('=>', $line);
        $key = $this->resolveKeyOrValue($key);
        $value = $this->resolveKeyOrValue($value);

        // Check if we are at the start of a nested array
        if (str_starts_with($value, '[')) {
            $tmpKeyPrefixes[] = $key;

            return [];
        }

        // Build the key
        if (count($tmpKeyPrefixes) > 0) {
            // Add extra dot to the key if it is a nested array
            $key = implode('.', $tmpKeyPrefixes) . ".{$key}";
        } else {
            $key = implode('.', $tmpKeyPrefixes) . "{$key}";
        }

        return [
            'key' => $key,
            'value' => $value,
        ];
    }
}
