<?php

namespace Flex\TranslationManager\Actions;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Str;

class importTranslationsAction
{
    private array $logs;

    private string $languagePath;

    public function __construct(?string $languagePath = null)
    {
        $this->languagePath = $languagePath ?? lang_path();
        $this->logs = [
            'success' => [],
            'errors' => [],
        ];
    }

    public function execute($command = null, $commandOutput = null): array
    {
        $directories = File::directories($this->languagePath);

        if ($commandOutput) {
            $progressBar = $commandOutput->createProgressBar(count($directories));
            $progressBar->start();
        }

        foreach ($directories as $directory) {
            $languageModel = $this->getOrCreateLanguage($directory);

            $this->syncingTranslation($directory, $languageModel);

            if ($commandOutput) {
                $progressBar->advance();
            }
        }

        if ($commandOutput) {
            $progressBar->finish();
        }

        return $this->logs;
    }

    private function getOrCreateLanguage(mixed $directory)
    {
        $folderName = basename($directory);
        $languageModelClass = Config::get('translation-manager.translation_language_model');

        return $languageModelClass::firstOrCreate(['folder_name' => $folderName], [
            'name' => $folderName,
            'folder_name' => $folderName,
        ]);
    }

    protected function syncingTranslation(string $languageDirectory, $languageModel): void
    {
        $languageFolderName = Str::afterLast($languageDirectory, '/');

        $files = File::allFiles($languageDirectory);

        foreach ($files as $file) {
            try {
                // only sync php files
                if ($file->getExtension() !== 'php') {
                    continue;
                }
                $fileName = $file->getBasename('.php');
                $tmpKeyPrefixes = [];

                $translationArray = require $file->getPathname();

                if (! is_array($translationArray) || count($translationArray) === 0) {
                    continue;
                }

                $this->importTranslationsRecursively($translationArray, $fileName, $languageModel);

                $this->logs['success'][$languageFolderName][] = $file->getFilename();

                // Delete translations that are no longer in the language files
                $translations = $languageModel->translations()->get();
                foreach ($translations as $translation) {
                    $key = "{$translation->file_name}.{$translation->key}";
                    $locale = $languageModel->folder_name;
                    if (! Lang::has($key, $locale)) {
                        $translation->delete();
                    }
                }
            } catch (\Exception $e) {
                // TODO: what to do with this exception? maybe log it for the developer?
                $this->logs['errors'][$languageFolderName][] = $file->getFilename();
            }
        }
    }

    private function importTranslationsRecursively($data, $fileName, $languageModel): void
    {
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $data = Arr::dot($value);
                foreach ($data as $nestedKey => $nestedValue) {
                    if (is_array($nestedValue)) {
                        if (count($nestedValue) === 0) {
                            continue;
                        }
                        $this->importTranslationsRecursively($nestedValue, $fileName, $languageModel);
                    } else {
                        $newKey = "{$key}.{$nestedKey}";
                        $this->addOrUpdateTranslation($newKey, $nestedValue, $fileName, $languageModel);
                    }
                }
            } else {
                $this->addOrUpdateTranslation($key, $value, $fileName, $languageModel);
            }
        }
    }

    private function addOrUpdateTranslation($key, $value, $fileName, $languageModel): void
    {
        // Check if key is empty otherwise updateOrCreate translation
        if (empty($key)) {
            return;
        }

        $languageModel->translations()->updateOrCreate(
            [
                'key' => $key],
            [
                'value' => $value,
                'file_name' => $fileName,
            ]
        );
    }
}
