<?php

namespace Flex\TranslationManager\Http\Controllers\Api;

use Flex\TranslationManager\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class LanguagesController extends Controller
{
    public function index(Request $request)
    {
        $languageQuery = config('translation-manager.translation_language_model')::query();
        if ($request->has('search')) {
            $languageQuery->where('name', 'like', '%'.$request->get('search').'%');
            $languageQuery->orWhere('folder_name', 'like', '%'.$request->get('search').'%');
        }

        $data = $languageQuery->get()
            ->map(function ($language) {
                return [
                    'id' => $language->id,
                    'name' => $language->name,
                    'folder_name' => $language->folder_name,
                    'is_default' => $language->folder_name === config('app.locale'),
                ];
            });

        return response()->json($data);
    }

    public function store(Request $request)
    {
        try {
            $requestProperties = $request->validate([
                'name' => 'required|string',
                'folder_name' => 'required|string',
            ]);

            config('translation-manager.translation_language_model')::query()
                ->create($requestProperties);

            return response()->json();
        } catch (ValidationException $e) {
            report($e);

            return response()->json([
                'message' => $e->validator->getMessageBag()->getMessages(),
            ], 422);
        } catch (\Exception $e) {
            report($e);

            return response()->json([
                'message' => 'Error while creating language',
            ], 422);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $requestProperties = $request->validate([
                'name' => 'required|string',
                'folder_name' => 'required|string',
            ]);

            $languageModel = config('translation-manager.translation_language_model')::query()
                ->findOrFail($id);

            $languageModel->update($requestProperties);

            return response()->json();
        } catch (ValidationException $e) {
            report($e);

            return response()->json([
                'message' => $e->validator->getMessageBag()->getMessages(),
            ], 422);
        } catch (\Exception $e) {
            report($e);

            return response()->json([
                'message' => 'Error while updating language',
            ], 422);
        }
    }

    public function destroy($id)
    {
        try {
            $languageModel = config('translation-manager.translation_language_model')::query()
                ->findOrFail($id);

            $languageModel->delete();

            return response()->json();
        } catch (\Exception $e) {
            report($e);

            return response()->json([
                'message' => 'Error while deleting language',
            ], 422);
        }
    }
}
