<?php

namespace Flex\TranslationManager\Http\Controllers\Api;

use Flex\TranslationManager\Http\Requests\StoreUpdateTranslationRequest;
use Flex\TranslationManager\Service\TranslationService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class TranslationController
{
    public function index(Request $request): JsonResponse
    {
        $requestProperties = $request->validate([
            'locale' => ['required', 'string'],
        ]);

        $translationService = new TranslationService();

        $locale = $requestProperties['locale'];

        $languageModel = config('translation-manager.translation_language_model');
        $languageModel = $languageModel::query()->where('folder_name', $locale)->firstOrFail();

        $translations = $languageModel
            ->translations()
            ->get()
            ->map(function ($translation) use ($translationService) {
                return [
                    'id' => $translation->id,
                    'key' => $translation->key,
                    'display_key' => $translationService->generateDisplayKey($translation->file_name, $translation->key),
                    'value' => $translation->value,
                    'file_name' => $translation->file_name,
                ];
            });

        $data = [
            'language' => $languageModel,
            'translations' => $translations,
        ];

        return response()->json($data);
    }

    public function store(Request $request): JsonResponse
    {
        try {
            $fileNameRequired = config('translation-manager.separate_translation_files') ?? true;
            $validateProperties = [
                'locale' => ['required', 'string'],
                'translation_key' => ['required', 'string'],
                'translation_value' => ['required', 'string'],
                'file_name' => ['nullable', 'string'],
            ];

            $requestProperties = $request->validate($validateProperties);

            $languageModel = config('translation-manager.translation_language_model');
            $translationModel = config('translation-manager.translation_model');

            $languageModel = $languageModel::query()->where('folder_name', $requestProperties['locale'])->firstOrFail();

            $translationModel::query()->updateOrCreate([
                'translation_language_id' => $languageModel->id,
                'key' => $requestProperties['translation_key'],
                'file_name' => $requestProperties['file_name'] ?? null,
            ], [
                'value' => $requestProperties['translation_value'],
            ]);

            return response()->json();
        } catch (ValidationException $e) {
            return response()->json([
                'message' => $e->validator->getMessageBag()->getMessages(),
            ], 422);
        } catch (\Exception $e) {
            report($e);

            return response()->json([
                'message' => 'Error creating translation.',
            ], 422);
        }
    }

    public function update(StoreUpdateTranslationRequest $request): JsonResponse
    {
        try {
            $fileNameRequired = config('translation-manager.separate_translation_files');
            $requestProperties = $request->validated();
            $languageModel = config('translation-manager.translation_language_model');
            $translationModel = config('translation-manager.translation_model');

            $languageModel = $languageModel::query()->where('folder_name', $requestProperties['locale'])->firstOrFail();

            $translation = $translationModel::query()->updateOrCreate([
                'translation_language_id' => $languageModel->id,
                'key' => $requestProperties['translation_key'],
                'file_name' => $fileNameRequired ? $requestProperties['file_name'] : null,
            ], [
                'value' => $requestProperties['translation_value'],
            ]);

            return response()->json([
                'id' => $translation->id,
            ]);
        } catch (\Exception $e) {
            report($e);

            return response()->json([
                'message' => 'Error updating translation.',
            ], 422);
        }
    }

    public function destroy($id): JsonResponse
    {
        try {
            $translationModel = config('translation-manager.translation_model');

            $translationModel::query()->findOrFail($id)->delete();

            return response()->json();
        } catch (\Exception $e) {
            report($e);

            return response()->json([
                'message' => 'Error deleting translation.',
            ], 422);
        }
    }
}
