<?php

namespace Flex\TranslationManager\Http\Controllers\Api;

use Flex\TranslationManager\Events\PublishedTranslationsEvent;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;

class PublishTranslationController
{
    public function store()
    {
        try {
            $useSeparateFiles = config('translation-manager.separate_translation_files');
            $backupTranslation = config('translation-manager.backup_count') > 0;

            if ($backupTranslation) {
                $this->backupLangFolder();
            }

            if ($useSeparateFiles) {
                $this->publishSeparateFiles();
            } else {
                $this->publishSingleFile();
            }

            event(new PublishedTranslationsEvent());

            return response()->json();
        } catch (\Exception $e) {
            report($e);

            return response()->json([
                'message' => 'Error publishing translations.',
            ], 422);
        }
    }

    private function publishSingleFile()
    {
        $languageModelClass = config('translation-manager.translation_language_model');
        foreach ($languageModelClass::query()->get() as $language) {
            $translations = $language->translations()->get()->mapWithKeys(function ($translation, $removeFileName) {
                if (true) {
                    return [$translation->key => $translation->value];
                }
                //                return ["{$translation->file_name}.{$translation->key}" => $translation->value];
            })->toArray();

            $fileName = $language->name;
            $this->generateFile($language, $fileName, $translations);
        }
    }

    private function publishSeparateFiles()
    {
        $languageModelClass = config('translation-manager.translation_language_model');
        foreach ($languageModelClass::query()->get() as $language) {
            $groupedTranslations = $language->translations()->get()->groupBy('file_name');
            foreach ($groupedTranslations as $fileName => $translationValues) {
                $translations = [];
                foreach ($translationValues as $translation) {
                    // Support nested keys
                    if (str_contains($translation->key, '.')) {
                        Arr::set($translations, $translation->key, $translation->value);
                    } else {
                        $translations[$translation->key] = $translation->value;
                    }
                }
                $this->generateFile($language, $fileName, $translations);
            }
        }
    }

    /**
     * @throws \Exception
     */
    private function generateFile(mixed $language, int|string $fileName, $translations)
    {
        $path = lang_path($language->folder_name);
        $filePath = $path.'/'.$fileName.'.php';
        if (! file_exists($path)) {
            mkdir($path, 0777, true);
        }

        $translations = collect($translations)->map(function ($value, $key) {
            return $this->resolveSpecialCharacters($value);
        })->toArray();

        $translationArray = var_export($translations, true);

        // Change array() to []
        $translationArray = str_replace('array (', '[', $translationArray);
        $translationArray = str_replace(')', ']', $translationArray);

        // Replace triple backslash with one backslash
        $translationArray = str_replace('\\\\\\', '\\', $translationArray);

        $data = '<?php'.PHP_EOL.'return '.$translationArray.';';
        file_put_contents($filePath, $data);
    }

    private function resolveSpecialCharacters($value)
    {
        if (is_string($value)) {
            if (preg_match('/[\'^£$%&*()}{@#~?><>,|=_+¬-]/', $value)) {
                $value = Str::of($value)->replace("'", "\\'")->value();
                if (Str::startsWith($value, '"')) {
                    $value = Str::replaceFirst('"', '', $value);
                }
                if (Str::endsWith($value, '"') || Str::endsWith($value, '",')) {
                    $value = Str::replaceLast('",', '', $value);
                    $value = Str::replaceLast('"', '', $value);
                }
            }
        } else {
            if (is_array($value)) {
                $value = collect($value)->map(function ($value, $key) {
                    return $this->resolveSpecialCharacters($value);
                })->toArray();
            }
        }

        return $value;
    }

    private function backupLangFolder(): void
    {
        $backupPath = storage_path('app/lang_backup');
        if (! file_exists($backupPath)) {
            mkdir($backupPath, 0777, true);
        }

        // check if backup folder has more than 10 backups
        $files = scandir($backupPath);

        $maxBackupCount = config('translation-manager.backup_count') + 1;
        if (count($files) > $maxBackupCount) {
            $oldestFile = $files[2];
            // delete oldest backup
            $oldestBackup = $backupPath.'/'.$oldestFile;
            File::deleteDirectory($oldestBackup);
        }
        $backupPath .= '/'.date('Y-m-d_H-i-s');
        mkdir($backupPath, 0777, true);

        $this->recurseCopy(lang_path(), $backupPath);
    }

    private function recurseCopy(string $lang_path, string $backupPath)
    {
        $dir = opendir($lang_path);
        while (false !== ($file = readdir($dir))) {
            if (($file != '.') && ($file != '..')) {
                if (is_dir($lang_path.'/'.$file)) {
                    $this->recurseCopy($lang_path.'/'.$file, $backupPath.'/'.$file);
                } else {
                    $fromPath = $lang_path.'/'.$file;
                    $toPath = $backupPath.'/'.$file;
                    if (! file_exists($backupPath)) {
                        mkdir($backupPath, 0777, true);
                    }
                    copy($fromPath, $toPath);
                }
            }
        }
        closedir($dir);
    }
}
